import type { Metadata } from "next";
import { notFound } from "next/navigation";
import { setRequestLocale } from "next-intl/server";
import { Navbar } from "@/components/navbar";
import { Footer } from "@/modules/landing/components";
import { getProjectBySlug, getProjects, getStrapiMediaUrl } from "@/core/lib/api";
import { buildMetadataFromSeo } from "@/core/lib/seo";
import { ProjectDetailsContent } from "./project-details-content";

type Props = {
    params: Promise<{ locale: string; slug: string }>;
};

export async function generateMetadata({ params }: Props): Promise<Metadata> {
    const { locale, slug } = await params;
    const project = await getProjectBySlug(slug, { locale });

    if (!project) {
        return buildMetadataFromSeo({
            locale: (locale === "ar" ? "ar" : "en") as "en" | "ar",
            seo: {
                metaTitle: "Project Not Found | YS Lootah Tech",
                metaDescription:
                    "The requested project could not be found. Explore YS Lootah Tech case studies and digital solutions across industries.",
            },
        });
    }

    const cover = project.cover_image;

    return buildMetadataFromSeo({
        seo: (project as any).seo ?? undefined,
        openGraph: (project as any).openGraph ?? undefined,
        locale: (locale === "ar" ? "ar" : "en") as "en" | "ar",
        override: {
            title: `${project.title} | YS Lootah Tech`,
            openGraph: {
                images: cover
                    ? [
                        {
                            url: getStrapiMediaUrl(cover.url),
                            alt: cover.alternativeText || project.title,
                        },
                    ]
                    : undefined,
            },
        },
    });
}

export async function generateStaticParams() {
    const projects = await getProjects();
    return projects.map((project) => ({
        slug: project.slug,
    }));
}

export default async function ProjectPage({ params }: Props) {
    const { locale, slug } = await params;
    setRequestLocale(locale);

    const project = await getProjectBySlug(slug, { locale });

    if (!project) {
        notFound();
    }

    return (
        <div className="min-h-screen bg-slate-950">
            <Navbar />
            <ProjectDetailsContent project={project} />
            <Footer />
        </div>
    );
}